// *****************************************************************************
/**
 * @brief JavaScript functions for displaying tooltips.
 * @file tooltips.js
 * @date 4 may 2012
 * @author Mathieu BRUNOT (mb.mathieu.brunot@gmail.com)
 * 
 * @version 1.2
 * @package js
 * 
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * 
 * JavaScript functions for displaying tooltips.
 * 
 * It is highly advised to use jQuery to allow an easier use of tooltips.
 * Using jQuery, you will only have to include this file into your page and
 * use the title attribute of HTML tags to generate tooltips.
 * The jQuery version of this file automatically parse the page when it's 
 * ready and replace your titles with custom titles.
 * This way, it's easier for you to write your page and the default tooltips 
 * will still be working even if you disable JavaScript !
 * 
 * The non-jQuery version does not allow such simple use...
 * 
 * @see http://jquery.com/
 * 
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @internal To Do List : 
 *     @todo ER#001 | Medium
 *     	Disable when in print mode: we lose the title attribute (which could be
 *     	used for print mode) while it will never be displayed in print mode.
 *     	Look up jmediatype for current media type detection.
 *     	@see http://code.google.com/p/jmediatype/
 * 
 *     @todo ER/PR#xxx | Major/Medium/Minor
 *     	...
 * 
 */
/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * ChangeLog : 
 * 
 * CRT | brunotm | 4 may 2012 
 * 		File creation
 * 
 * ENH | brunotm | 14 may 2012 
 * 		jQuery auto tooltip parser
 * 
 * ENH | brunotm | 24 may 2012 
 * 		Adapt position if close to the border of the window
 * 
 * CRT/UPD/ENH/BUG/CHG | author | date 
 * 		What has been done...
 * 
 */
/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * MIT License
 * 
 * Copyright (C) 2012 Mathieu BRUNOT
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a 
 * copy of this software and associated documentation files (the "Software"), 
 * to deal in the Software without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, 
 * and/or sell copies of the Software, and to permit persons to whom the 
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 * 
 */
// *****************************************************************************

// if jQuery is available
if (typeof jQuery !== 'undefined') {
	
	/**
	 * @brief Function to automatically parse the page and generate 
	 * 	custom tooltips.
	 * 
	 * @param $ jQuery instance
	 * 
	 * For every element that has a title, we change the tooltip to a 
	 * custom-made tooltip, using jQuery and CSS.
	 * The tooltip will follow the mouse has long as it is above an element 
	 * with a title.
	 * To ensure that the browser will not display the default tooltip, we 
	 * get rid of the title of the element.
	 * 
	 * The tooltip uses the 'tooltips' CSS class. Edit it to change the 
	 * aspect of your tooltip.
	 * 
	 * TODO Disable when in print mode (look up jmediatype)
	 * 
	 * @author brunotm
	 * @see http://jquery.com/
	 */
	(function autoTooltip($) {
		// When the document is ready, launch the core of the function
		$(document).ready(function() {
			// For each DOM of the page that has a title
			$('body [title]').each(function() {
				var $this = $(this);
				var $tooltip;
				
				// Make sure the item has a title
				if( $this.attr('title').length>0 ) {
					var title = this.title;
					
					// Empty the title
					this.title = '';
					
					// Actions to be taken when hovering
					$this.hover(function(e) {
						
						// Build the tooltip and append it to the body
						$tooltip = $('<div id="tooltip" class="tooltips"/>')
						.appendTo('body')
						.hide();
						
						// Fill the content of the tooltip
						$tooltip.html(title);
						
						// Calculate the best position given the size of the window
						var top = e.pageY+10;
						if (top+$tooltip.height() >= $(window).height())
							top = e.pageY - $tooltip.height() -10;
						
						var left = e.pageX+10;
						if (left+$tooltip.width() >= $(window).width() -10)
							left = e.pageX  - $tooltip.width() - 20;
						
						// Set the tooltip position and fade it in
						$tooltip.css({top: top, left: left}).fadeIn(350);
					}, function() {
						// Remove the tooltip
						$tooltip.remove();
					});
					
					// Bind a mouse move function
					$this.mousemove(function(e) {
						//$tooltip = $('#tooltip');
						// Calculate the best position given the size of the window
						var top = e.pageY+10;
						if (top+$tooltip.height() >= $(window).height())
							top = e.pageY - $tooltip.height() -10;
						
						var left = e.pageX+10;
						if (left+$tooltip.width() >= $(window).width() -10)
							left = e.pageX  - $tooltip.width() -20;
						
						// Move the tooltip relatively to the mouse
						$tooltip.css({top: top, left: left});
					});
				}
			});
		});
	})(jQuery);
	
}
//*****************************************************************************
//if jQuery is NOT available
else {
	/**
	 * @brief Lesser function/class for handling tooltips
	 * @returns instance of tooltip handling
	 * 
	 * This lesser version does not use directly the title attribute of HTML 
	 * tags.
	 * 
	 * To use this version, you will have to write tags this way:
	 * <a onmouseover="javascript: tooltip.show('Tooltip content');"
	 * 		onmouseout="javascript: tooltip.hide();">Over Me!</span>
	 * 
	 */
	function tooltip() {
		/**
		 * @brief DOM div which we will use for the tooltips
		 * @param text content of the tooltip
		 */
		this.div = null;
		
		/**
		 * @brief Function to show the tooltip
		 * @param text content of the tooltip
		 */
		this.show = function (text) {
			if (this.div == null) {
				// add tooltips to body
				this.div = document.createElement('div');
				this.div.id = 'tooltip';
				this.div.className = 'tooltips';
				this.div.style.visibility = "hidden";
				document.body.appendChild(this.div);
				
				// Make tooltips visible and fill with the text
				this.div.innerHTML = text;
				this.div.style.visibility = "visible";
				this.isVisible = true;
			}
		};
		
		/**
		 * @brief Function to hide the tooltip
		 */
		this.hide = function () {
			// Hide tooltips
			if (this.div != null) {
				this.div.style.visibility = "hidden";
			
				// Remove tooltips from body
				document.body.removeChild(this.div);
				this.div = null;
			}
		};
		
		/**
		 * @brief Function to bind the move of the mouse to a function
		 * 
		 * @param mouseMoveHandler handler of the mouse move event
		 * 
		 * The function moveHandler must receive an event.
		 */
		this.bindMove = function (mouseMoveHandler) {
			document.onmousemove = mouseMoveHandler;
		};
	}
	
	var tooltip = new tooltip();
	
	/**
	 * @brief Function to handle movements of the mouse
	 * @param e mouse move event
	 * 
	 * The tooltip will follow the mouse as long as it must be shown.
	 * 
	 */
	function moveTooltip(e) {
		if (tooltip.div) {
			if (navigator.appName != "Microsoft Internet Explorer") {
				tooltip.div.style.top = e.pageY + 10 + "px";
				tooltip.div.style.left = e.pageX + 5 + "px";
			} else {
				if (document.documentElement.clientWidth > 0) {
					tooltip.div.style.top = 10 + e.y
							+ document.documentElement.scrollTop + "px";
					tooltip.div.style.left = 20 + e.x
							+ document.documentElement.scrollLeft + "px";
				} else {
					tooltip.div.style.top = 10 + e.y
							+ document.body.scrollTop + "px";
					tooltip.div.style.left = 20 + e.x
							+ document.body.scrollLeft + "px";
				}
			}
		}
	}
	
	tooltip.bindMove(moveTooltip);
}

//*****************************************************************************